import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Box, User, Calendar, Star, Code, TrendingUp, Award, CheckCircle2, Layers } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyProjects.css'

const MyProjects = () => {
  const [projects, setProjects] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('model3dProjects')
    if (saved) {
      try {
        setProjects(JSON.parse(saved))
      } catch (e) {
        setProjects([])
      }
    } else {
      const defaultProjects = [
        {
          id: 1,
          course: 'Blender для начинающих',
          instructor: 'Алексей Волков',
          submissionDate: '2025-03-28',
          rating: 5,
          progress: 85
        },
        {
          id: 2,
          course: 'Maya Advanced',
          instructor: 'Мария Соколова',
          submissionDate: '2025-03-25',
          rating: 5,
          progress: 92
        },
        {
          id: 3,
          course: '3ds Max Профессионал',
          instructor: 'Дмитрий Новиков',
          submissionDate: '2025-03-22',
          rating: 4,
          progress: 78
        }
      ]
      setProjects(defaultProjects)
      localStorage.setItem('model3dProjects', JSON.stringify(defaultProjects))
    }
  }, [])

  const totalProjects = projects.length
  const averageRating = projects.length > 0
    ? (projects.reduce((sum, p) => sum + p.rating, 0) / projects.length).toFixed(1)
    : 0
  const totalProgress = projects.length > 0
    ? Math.round(projects.reduce((sum, p) => sum + p.progress, 0) / projects.length)
    : 0
  const completedProjects = projects.filter(p => p.progress === 100).length

  return (
    <div className="my-projects">
      <div className="container">
        <motion.section 
          className="hero-clean"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="hero-clean-content">
            <h1 className="hero-clean-title">Мои проекты</h1>
            <p className="hero-clean-description">
              Отслеживайте прогресс ваших 3D-проектов и достижения в обучении
            </p>
          </div>
        </motion.section>

        <motion.section 
          className="stats-clean"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <div className="stats-clean-grid">
            <motion.div 
              className="stat-clean-item"
              whileHover={{ y: -4, transition: { duration: 0.2 } }}
            >
              <div className="stat-clean-icon">
                <Box size={32} strokeWidth={2} color="#1A1F2E" />
              </div>
              <div className="stat-clean-content">
                <div className="stat-clean-value">{totalProjects}</div>
                <div className="stat-clean-label">Проектов</div>
              </div>
            </motion.div>
            <motion.div 
              className="stat-clean-item"
              whileHover={{ y: -4, transition: { duration: 0.2 } }}
            >
              <div className="stat-clean-icon">
                <Star size={32} strokeWidth={2} color="#1A1F2E" />
              </div>
              <div className="stat-clean-content">
                <div className="stat-clean-value">{averageRating}</div>
                <div className="stat-clean-label">Рейтинг</div>
              </div>
            </motion.div>
            <motion.div 
              className="stat-clean-item"
              whileHover={{ y: -4, transition: { duration: 0.2 } }}
            >
              <div className="stat-clean-icon">
                <TrendingUp size={32} strokeWidth={2} color="#1A1F2E" />
              </div>
              <div className="stat-clean-content">
                <div className="stat-clean-value">{totalProgress}%</div>
                <div className="stat-clean-label">Прогресс</div>
              </div>
            </motion.div>
            <motion.div 
              className="stat-clean-item"
              whileHover={{ y: -4, transition: { duration: 0.2 } }}
            >
              <div className="stat-clean-icon">
                <Award size={32} strokeWidth={2} color="#1A1F2E" />
              </div>
              <div className="stat-clean-content">
                <div className="stat-clean-value">{completedProjects}</div>
                <div className="stat-clean-label">Завершено</div>
              </div>
            </motion.div>
          </div>
        </motion.section>

        <ImageSlider images={[
          '/images/model-1.jpg',
          '/images/model-2.jpg',
          '/images/model-3.jpg',
          '/images/model-4.jpg'
        ]} />

        <motion.section 
          className="projects-clean"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.3 }}
        >
          <div className="section-clean-header">
            <h2 className="section-clean-title">Мои проекты</h2>
          </div>
          {projects.length > 0 ? (
            <div className="projects-clean-list">
              {projects.map((project, index) => (
                <motion.div
                  key={project.id}
                  className="project-clean-item"
                  initial={{ opacity: 0, x: -30 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: 0.4 + index * 0.1, duration: 0.5 }}
                  whileHover={{ x: 8, transition: { duration: 0.2 } }}
                >
                  <div className="project-clean-indicator"></div>
                  <div className="project-clean-content">
                    <div className="project-clean-header">
                      <div className="project-clean-badge">{project.course.split(' ')[0]}</div>
                      <div className="project-clean-rating">
                        {Array.from({ length: 5 }, (_, i) => (
                          <Star
                            key={i}
                            size={16}
                            fill={i < project.rating ? '#1A1F2E' : 'none'}
                            stroke={i < project.rating ? '#1A1F2E' : '#CBD5E0'}
                            strokeWidth={2}
                          />
                        ))}
                      </div>
                    </div>
                    <h3 className="project-clean-course">{project.course}</h3>
                    <div className="project-clean-details">
                      <div className="project-clean-detail">
                        <User size={16} strokeWidth={2} color="#1A1F2E" />
                        <span>{project.instructor}</span>
                      </div>
                      <div className="project-clean-detail">
                        <Calendar size={16} strokeWidth={2} color="#4A5568" />
                        <span>{new Date(project.submissionDate).toLocaleDateString('ru-RU', { day: 'numeric', month: 'long', year: 'numeric' })}</span>
                      </div>
                    </div>
                    <div className="project-clean-progress">
                      <div className="project-clean-progress-header">
                        <span className="progress-clean-label">Прогресс</span>
                        <span className="progress-clean-value">{project.progress}%</span>
                      </div>
                      <div className="project-clean-progress-bar">
                        <motion.div 
                          className="project-clean-progress-fill"
                          initial={{ width: 0 }}
                          animate={{ width: `${project.progress}%` }}
                          transition={{ delay: 0.6 + index * 0.1, duration: 1, ease: "easeOut" }}
                        />
                      </div>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-clean">
              <div className="empty-clean-icon">
                <Box size={56} strokeWidth={1.5} />
              </div>
              <h3 className="empty-clean-title">Начните свой путь в 3D</h3>
              <p className="empty-clean-text">Запишитесь на курс и увидите свой первый проект здесь</p>
            </div>
          )}
        </motion.section>

        <motion.section 
          className="features-clean"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.5 }}
        >
          <div className="features-clean-header">
            <h2 className="section-clean-title">Преимущества обучения</h2>
          </div>
          <div className="features-clean-grid">
            {[
              { icon: Layers, title: 'Структурированная программа', text: 'Пошаговое обучение от основ до продвинутых техник' },
              { icon: CheckCircle2, title: 'Практические проекты', text: 'Реальные задачи и портфолио для вашей карьеры' },
              { icon: Code, title: 'Обратная связь', text: 'Персональные консультации с опытными преподавателями' },
              { icon: Award, title: 'Сертификаты', text: 'Официальные документы о прохождении курсов' }
            ].map((feature, index) => {
              const IconComponent = feature.icon
              return (
                <motion.div
                  key={feature.title}
                  className="feature-clean-card"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.6 + index * 0.1, duration: 0.5 }}
                  whileHover={{ y: -4, transition: { duration: 0.2 } }}
                >
                  <div className="feature-clean-icon">
                    <IconComponent size={32} strokeWidth={2} color="#1A1F2E" />
                  </div>
                  <h3 className="feature-clean-title">{feature.title}</h3>
                  <p className="feature-clean-text">{feature.text}</p>
                  <div className="feature-clean-accent"></div>
                </motion.div>
              )
            })}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default MyProjects

